<?php
declare(strict_types=1);

function current_user(): ?array {
  return $_SESSION['user'] ?? null;
}

function require_login(): void {
  if (!current_user()) {
    redirect('login.php');
  }
}

function is_admin(): bool {
  $u = current_user();
  return $u && ($u['role'] ?? '') === 'admin';
}

function login(PDO $db, string $email, string $password): bool {
  $stmt = $db->prepare("SELECT u.id, u.email, u.full_name, r.name AS role, u.password_hash 
                        FROM users u JOIN roles r ON r.id=u.role_id 
                        WHERE u.email = ? AND u.is_active=1 LIMIT 1");
  $stmt->execute([strtolower(trim($email))]);
  $u = $stmt->fetch();
  if (!$u) return false;
  if (!password_verify($password, $u['password_hash'])) return false;

  session_regenerate_id(true);
  $_SESSION['user'] = [
    'id' => (int)$u['id'],
    'email' => $u['email'],
    'full_name' => $u['full_name'],
    'role' => $u['role'],
  ];

  audit($db, 'login', null, 'User logged in');
  return true;
}

function logout(PDO $db): void {
  audit($db, 'logout', null, 'User logged out');
  $_SESSION = [];
  if (ini_get("session.use_cookies")) {
    $params = session_get_cookie_params();
    setcookie(session_name(), '', time() - 42000,
      $params["path"], $params["domain"],
      $params["secure"], $params["httponly"]
    );
  }
  session_destroy();
}

function audit(PDO $db, string $action, ?int $matter_id, string $details=''): void {
  $u = current_user();
  $stmt = $db->prepare("INSERT INTO audit_logs (user_id, action, matter_id, details, ip_address, user_agent, created_at)
                        VALUES (?, ?, ?, ?, ?, ?, ?)");
  $stmt->execute([
    $u['id'] ?? null,
    $action,
    $matter_id,
    $details,
    client_ip(),
    user_agent(),
    now()
  ]);
}

function can_access_matter(PDO $db, int $matter_id): bool {
  $u = current_user();
  if (!$u) return false;
  if (($u['role'] ?? '') === 'admin') return true;

  $stmt = $db->prepare("SELECT 1 FROM matter_users WHERE matter_id=? AND user_id=? LIMIT 1");
  $stmt->execute([$matter_id, $u['id']]);
  return (bool)$stmt->fetchColumn();
}
